<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Venta extends Model
{
    use HasFactory;

    protected $table = 'ventas';

    protected $fillable = [
        'sucursal_id',
        'sucursal_id',
        'cliente_id',
        'timbrador_id',
        'total',
        'subtotal',
        'iva',
        'ieps',
        'estatus',
        'cancelado',
        'fecha_registro',
        'observacion_cancelado',
        'direccion_id',
        'datos_fiscales_id',
        'opcion_pago_transferencia_id',
        'comprobante_pago',
        'es_pago_tienda',
        'es_tienda_online',
    ];

    protected $appends = ['comprobante_pago_path'];

    public function cliente(): BelongsTo
    {
        return $this->belongsTo(Cliente::class,'cliente_id');
    }

    public function sucursal(): BelongsTo
    {
        return $this->belongsTo(Sucursal::class, 'sucursal_id');
    }

    public function ventaDetalle(): HasMany
    {
        return $this->hasMany(VentaDetalle::class, 'venta_id');
    }

    public function ventaSesion(): HasMany
    {
        return $this->hasMany(VentaSesion::class, 'venta_id');
    }

    public function pagos(): HasMany
    {
        return $this->hasMany(Pago::class, 'venta_id');
    }

    public function opcionPagoTransferencia()
    {
        return $this->belongsTo(OpcionPagoTransferencia::class, 'opcion_pago_transferencia_id');
    }

    public function direccion(): HasOne
    {
        return $this->hasOne(DireccionEntrega::class, 'id', 'direccion_id');
    }

    public function datosFiscales(): HasOne  
    {
        return $this->hasOne(DatoFiscal::class, 'id', 'datos_fiscales_id');
    }
    
    public function vale(): HasOne
    {
        return $this->hasOne(VentaDetalle::class, 'venta_id', 'id')->where('vale_id', '!=', null);
    }

    public function getComprobantePagoPathAttribute()
    {
        return asset(str_replace("public", "storage", $this->comprobante_pago));
    }

    public function envio(): HasOne
    {
        return $this->hasOne(Envio::class, 'venta_id', 'id')->selectRaw("
            *,
            date_format(envios.fecha_envio,'%d-%m-%Y') fecha_envio_custom,
            date_format(envios.fecha_entrega_estimada,'%d-%m-%Y') fecha_entrega_estimada_custom
        ");
    }
}
