<?php

namespace App\Http\Controllers;

use App\Models\Tratamiento;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class TratamientoController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request): JsonResponse
    {
        $ids=[];
        if(isset($request->tratamientos)){
            $ids=array_column($request->tratamientos,'id');
        }
        if(isset($request->filtro)) {
            $ids=array_column($request->filtro,'tratamiento_id');
            foreach ($ids as $key => $value) {
                if(is_null($value)) {
                    unset($ids[$key]);
                }
            }
        }
        $tratamiento=Tratamiento::selectRaw('
            tratamientos.*,
            (iva/100) iva_def,
            (ieps/100) ieps_def,
            (tratamientos.precio*(iva/100)) iva,
            (tratamientos.precio + (tratamientos.precio*(iva/100))) total
        ')
        ->where(['estatus'=>'activo'])->whereNotIn('id',$ids)->get();
        return response()->json([
            'data' =>$tratamiento,
            'id' =>$ids,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'tratamiento' => ['required', 'string'],
            'precio' => ['required', 'numeric', 'max:999999999.99'],
            'tiene_iva' => ['required', 'max:2'],
            'iva' => ['required', 'numeric', 'max:99'],
            'tiene_ieps' => ['required','max:2'],
            'ieps' => ['required', 'numeric', 'max:99'],
            'duracion' => ['required'],
            'estatus' => ['required', 'string','max:10'],
            'tipo_tratamiento_id' => ['required', Rule::exists('tipos_tratamientos', 'id')]
        ]);

        $tratamiento = Tratamiento::create($request->only([
            'tratamiento',
            'precio',
            'tiene_iva',
            'iva',
            'tiene_ieps',
            'ieps',
            'desc_tratamiento',
            'estatus',
            'duracion',
            'tipo_tratamiento_id',
        ]));

        return response()->json([
            'data' => $tratamiento,
        ], JsonResponse::HTTP_OK);
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Tratamiento $tratamiento): JsonResponse
    {
        return response()->json([
            'data' => [
                'tratamiento'=>$tratamiento,
                'tipoTratamiento'=>$tratamiento->tipoTratamiento,
            ],
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Tratamiento $tratamiento): JsonResponse
    {
        $request->validate([
            'tratamiento' => ['required', 'string'],
            'precio' => ['required', 'numeric', 'max:999999999.99'],
            'tiene_iva' => ['required', 'max:2'],
            'iva' => ['required', 'numeric', 'max:99.99'],
            'tiene_ieps' => ['required','max:2'],
            'ieps' => ['required', 'numeric', 'max:99.99'],
            'duracion' => ['required'],
            'estatus' => ['required', 'string','max:10'],
            'tipo_tratamiento_id' => ['required', Rule::exists('tipos_tratamientos', 'id')]
        ]);

        $tratamiento->update($request->only([
            'tratamiento',
            'precio',
            'tiene_iva',
            'iva',
            'tiene_ieps',
            'ieps',
            'desc_tratamiento',
            'estatus',
            'duracion',
            'tipo_tratamiento_id',
        ]));

        return response()->json([
            'data' => $tratamiento,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Tratamiento $tratamiento): JsonResponse
    {
        // if (\count($tratamiento->sesionTratamiento) > 0 || \count($tratamiento->ventasDetalles) > 0) {
        // } else {
        //     $tratamiento->delete();
        // }
        $tratamiento->update([
            'estatus' => 'inactivo',
        ]);

        return response()->json(null, JsonResponse::HTTP_NO_CONTENT);
    }
}
