<?php

namespace App\Http\Controllers;

use App\Mail\AvisoActivacionCita;
use App\Mail\AvisoNuevaCita;
use App\Models\Cliente;
use App\Models\Sesion;
use App\Models\SesionTratamiento;
use App\Models\Sucursal;
use App\Models\User;
use App\Models\ValeSesion;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use PhpParser\Node\Expr\Cast\Object_;
use Illuminate\Support\Facades\Mail;

class SesionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request): JsonResponse
    {
        $sucursal_id = null;
        if (count($request->user()->configUsersSucursales) > 0) {
            $sucursal_id = $request->user()->configUsersSucursales[0]->sucursal_id;
        }

        $sucursal_id = isset($request->sucursal_id) ? $request->sucursal_id : $sucursal_id;
        $cliente = isset($request->cliente) ? $request->cliente : null;
        $fecha_inicio = isset($request->fecha_inicio) ? $request->fecha_inicio : null;
        $fecha_fin = isset($request->fecha_fin) ? $request->fecha_fin : null;
        $filtro_tratamiento = isset($request->filtro_tratamiento) ? $request->filtro_tratamiento : null;
        $filtro_terapista = isset($request->filtro_terapista) ? $request->filtro_terapista : null;

        $fecha_actual = date("Y-m-d");
        $fecha_activa = $fecha_actual;

        if (empty($fecha_inicio) || empty($fecha_fin)) {
            $fecha_inicio = date("Y-m", strtotime($fecha_actual . "- 1 months")) . '-01 00:00:00';
            $fecha_fin    = date("Y-m", strtotime($fecha_actual . "+ 1 months")) . '-15 23:59:00';
        } else {
            $fecha_activa = date("Y-m-d", strtotime($fecha_inicio . "+ 15 days"));
            $fecha_inicio = date("Y-m-d", strtotime($fecha_inicio . "- 15 days")) . ' 00:00:00';
            $fecha_fin    = date("Y-m-d", strtotime($fecha_fin . "+ 7 days")) . ' 23:59:00';
        }

        $sucursales = Sucursal::where(['estatus' => 'activo'])->select(['id','nombre'])->get();
        $terapistas = User::role('Terapista')->where(['estatus' => 'activo'])->select(['id','name'])->get();

        $sql = "
            select
                distinct
                date(fecha_asignada) fecha_asignada,
                count(distinct s.id) total_sesiones
            from sesiones s
            INNER JOIN clientes c on s.cliente_id=c.id
            INNER JOIN sucursales sc on s.sucursal_id=sc.id
            INNER JOIN users te on s.terapista_id=te.id
            INNER JOIN sesiones_tratamientos tre on s.id=tre.sesion_id
            INNER JOIN tratamientos tra on tre.tratamiento_id=tra.id
            WHERE fecha_asignada BETWEEN '" . $fecha_inicio . "' and '" . $fecha_fin . "'
        ";

        if (!$request->user()->hasRole('Cliente')) {
            $sql .= "
                and s.aprobado='SI'
            ";
        } else {
            $sql .= "
                and s.cliente_id=".$request->user()->cliente_id."
            ";
        }

        if ($sucursal_id != null && !empty($sucursal_id)) {
            $sql .= "
                and sc.id=" . $sucursal_id . "
            ";
        }

        if ($cliente != null && !empty($cliente)) {
            $sql .= "
                and lower(c.razonSocial) like lower('%" . $cliente . "%')
            ";
        }

        if ($filtro_terapista != null && !empty($filtro_terapista)) {
            $sql .= "
                and s.terapista_id=" . $filtro_terapista . "
            ";
        }

        if (!empty($filtro_tratamiento)) {
            $sql = $sql . "
                and lower(tra.tratamiento) like '%" . strtolower($filtro_tratamiento) . "%'
            ";
        }
        $sql .= "
            GROUP BY date(fecha_asignada)
            ORDER BY fecha_asignada asc

        ";

        $sesiones = DB::select($sql);

        $horarios = DB::select("SELECT MIN(hora_entrada) min, MAX(hora_salida) max, date(NOW()) fecha_limite FROM horarios");

        $eventos = [];
        foreach ($sesiones as $model_session) {
            $color = 'bg-primary';
            $eventos[] = (object)[
                'id' => $model_session->fecha_asignada,
                'title' => "[{$model_session->total_sesiones}] Ver citas",
                'start' => $model_session->fecha_asignada,
                'horarios' => $horarios,
                'className' => $color
            ];
        }

        if ($request->user()->hasRole('Cliente')) {
            $fecha_inicial = new \DateTime($fecha_inicio);
            $fecha_final = new \DateTime($fecha_actual);

            $intervalo = \DateInterval::createFromDateString('1 day');
            $periodo = new \DatePeriod($fecha_inicial , $intervalo, $fecha_final);

            foreach ($periodo as $dt) {
                $color = 'bg-danger';
                $eventos[] = (object)[
                    'id' => $dt->format("Y-m-d").'temp',
                    'title' => "No disponible",
                    'start' => $dt->format("Y-m-d"),
                    'className' => $color
                ];
            }
        }

        return response()->json([
            'data' => [
                'eventos'=>$eventos,
                'sucursales'=>$sucursales,
                'terapistas'=>$terapistas,
                'fecha_inicio'=>$fecha_inicio,
                'fecha_fin'=>$fecha_fin,
                'fecha_activa'=>$fecha_activa,
                'horarios'=>$horarios,
            ],
        ], JsonResponse::HTTP_OK);
    }

    public function fecha(Request $request): JsonResponse
    {
        $sesiones=[];
        if(!empty($request->fecha)){
            $sucursal_id = isset($request->sucursal_id) ? $request->sucursal_id : null;
            $cliente = isset($request->cliente) ? $request->cliente : null;
            $filtro_tratamiento = isset($request->filtro_tratamiento) ? $request->filtro_tratamiento : null;
            $filtro_terapista = isset($request->filtro_terapista) ? $request->filtro_terapista : null;

            $sql ="
            select
                distinct
                s.id,
                fecha_asignada,
                concat(date_format(fecha_asignada,'%l %p'),' - ',date_format(time(ADDTIME(fecha_asignada,s.duracion)),'%l %p')) horas,
                c.razonSocial cliente,
                c.celular,
                sc.nombre sucursal,
                te.name terapista,
                asistencia,
                (
                    case
                        WHEN (date(fecha_asignada) > date(SUBDATE(NOW(),INTERVAL 1 DAY)) AND asistencia='SI') then '#4361ee'
                        WHEN (date(fecha_asignada) > date(SUBDATE(NOW(),INTERVAL 1 DAY)) AND asistencia='NO') then '#e2a03f'
                        WHEN (date(fecha_asignada) < date(SUBDATE(NOW(),INTERVAL 1 DAY)) AND asistencia_cumplida='SI') then '#1abc9c'
                        else '#e7515a'
                    end
                ) estatus_color,
                (
                    case
                        WHEN (s.aprobado='NO') then '#e7515a'
                        WHEN (date(fecha_asignada) < date(SUBDATE(NOW(),INTERVAL 1 DAY)) AND asistencia_cumplida='SI') then '#1abc9c'
                        else '#e2a03f'
                    end
                ) estatus_color_cliente,
                s.aprobado
            from sesiones s
            INNER JOIN clientes c on s.cliente_id=c.id
            INNER JOIN sucursales sc on s.sucursal_id=sc.id
            INNER JOIN users te on s.terapista_id=te.id
            INNER JOIN sesiones_tratamientos tre on s.id=tre.sesion_id
            INNER JOIN tratamientos tra on tre.tratamiento_id=tra.id
            WHERE date(fecha_asignada)= date('".$request->fecha."')
            ";

            if (!$request->user()->hasRole('Cliente')) {
                $sql .= "
                    and s.aprobado='SI'
                ";
            } else {
                $sql .= "
                    and s.cliente_id=".$request->user()->cliente_id."
                ";
            }


            if ($sucursal_id != null && !empty($sucursal_id)) {
                $sql .= "
                    and sc.id=" . $sucursal_id . "
                ";
            }

            if ($cliente != null && !empty($cliente)) {
                $sql .= "
                    and lower(c.razonSocial) like lower('%" . $cliente . "%')
                ";
            }

            if ($filtro_terapista != null && !empty($filtro_terapista)) {
                $sql .= "
                    and s.terapista_id=" . $filtro_terapista . "
                ";
            }

            if (!empty($filtro_tratamiento)) {
                $sql = $sql . "
                    and lower(tra.tratamiento) like '%" . strtolower($filtro_tratamiento) . "%'
                ";
            }
            $sql.= "
                ORDER BY fecha_asignada asc
            ";

            $sesiones = DB::select($sql);

        }
        return response()->json([
            'data' =>[
                'sesiones'=>$sesiones,
                'fecha'=>fechaLetras($request->fecha),
            ]
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request) : JsonResponse
    {
        $sql="
        SELECT
            DATE_FORMAT(hora_entrada,'%l %p') hora_entrada,
            DATE_FORMAT(SUBTIME(hora_salida,'00:00:00'),'%l %p') hora_salida,
            (hora_entrada <= TIME('".$request->fecha_asignada."') and SUBTIME(hora_salida,'00:00:00') >= TIME('".$request->fecha_asignada."')) validacion
        from horarios where dia=DAYOFWEEK('".$request->fecha_asignada."') and sucursal_id=".$request->sucursal_id."
        ";
        $buscador = DB::select($sql);
        if(isset($buscador[0])&&$buscador[0]->validacion==0){
            return response()->json([
                'message' => 'Horario invalido',
                'errors' => [
                    'fecha_asignada' => [
                        'Solo se permite un horario entre '.$buscador[0]->hora_entrada.' - '.$buscador[0]->hora_salida
                    ]
                ],
            ], JsonResponse::HTTP_UNPROCESSABLE_ENTITY);
        }

        $request->validate([
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')],
            'cliente_id' => ['required', Rule::exists('clientes', 'id')],
            'terapista_id' => ['required', Rule::exists('users', 'id')],
            'duracion' => ['required'],
            'fecha_asignada' => ['required', 'date'],
            'asistencia' => ['required'],
            'preferencia_terepista' => ['required'],
            'vale' => ['required'],
            'asistencia_cumplida' => ['required'],
            'usa_expediente' => ['required'],
            'vale_id'=> ['required_if:vale,SI'],
        ]);

        $params = $request->only([
            'sucursal_id',
            'cliente_id',
            'terapista_id',
            'duracion',
            'fecha_asignada',
            'observaciones',
            'observacion_cliente',
            'asistencia',
            'preferencia_terepista',
            'vale',
            'asistencia_cumplida',
            'usa_expediente',
            'vale_id'
        ]);
        $notificar = false;
        if(isset($request->aprobado))
        {
            $params['aprobado']=$request->aprobado;
            if($request->aprobado == 'NO') $notificar = true;
        }

        if($request->vale == 'SI') {
            $params['vale_id'] = $request->vale_id;
        }

        $sesiones = Sesion::create($params);

        // Crear relación vale-sesión si se usa vale
        if($request->vale == 'SI' && $request->vale_id) {
            ValeSesion::create([
                'vale_id' => $request->vale_id,
                'sesion_id' => $sesiones->id
            ]);
        }

        foreach ($request->tratamientos_vistas as $value) {
            SesionTratamiento::create([
                'sesion_id'=>$sesiones->id,
                'tratamiento_id'=>$value['id']
            ]);
        }

        $sesiones->cliente->update($request->only(['celular']));

        if($notificar){
            Mail::to(env('MAIL_FROM_ADDRESS'))->send(new AvisoNuevaCita($sesiones));
        }

        return response()->json([
            'data' => $sesiones,
        ], JsonResponse::HTTP_OK);
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Sesion $sesion): JsonResponse
    {
        return response()->json([
            'data' => [
                'sesion'=>$sesion,
                'cliente'=>$sesion->cliente,
                'sucursal'=>$sesion->sucursal,
                'terapista'=>$sesion->terapista,
                'tratamientos'=>$sesion->sesionTratamiento,
            ],
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Sesion $sesion): JsonResponse
    {
        $sql="
        SELECT
            DATE_FORMAT(hora_entrada,'%l %p') hora_entrada,
            DATE_FORMAT(SUBTIME(hora_salida,'00:00:00'),'%l %p') hora_salida,
            (hora_entrada <= TIME('".$request->fecha_asignada."') and SUBTIME(hora_salida,'00:00:00') >= TIME('".$request->fecha_asignada."')) validacion
        from horarios where dia=DAYOFWEEK('".$request->fecha_asignada."') and sucursal_id=".$request->sucursal_id."
        ";
        $buscador = DB::select($sql);
        if(isset($buscador[0])&&$buscador[0]->validacion==0){
            return response()->json([
                'message' => 'Horario invalido',
                'errors' => [
                    'fecha_asignada' => [
                        'Solo se permite un horario entre '.$buscador[0]->hora_entrada.' - '.$buscador[0]->hora_salida
                    ]
                ],
            ], JsonResponse::HTTP_UNPROCESSABLE_ENTITY);
        }

        $request->validate([
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')],
            'cliente_id' => ['required', Rule::exists('clientes', 'id')],
            'terapista_id' => ['required', Rule::exists('users', 'id')],
            'duracion' => ['required'],
            'fecha_asignada' => ['required', 'date'],
            'fecha_asignada' => ['required', 'date'],
            'asistencia' => ['required'],
            'preferencia_terepista' => ['required'],
            'vale' => ['required'],
            'asistencia_cumplida' => ['required'],
            'usa_expediente' => ['required'],
            // 'vale_id'=> ['required_if:vale,SI'],
        ]);

        $params = $request->only([
            'sucursal_id',
            'cliente_id',
            'terapista_id',
            'duracion',
            'fecha_asignada',
            'observaciones',
            'observacion_cliente',
            'asistencia',
            'preferencia_terepista',
            'vale',
            'asistencia_cumplida',
            'usa_expediente',
            'vale_id'
        ]);
        $notificar = false;
        if(isset($request->aprobado)) {
            $params['aprobado']=$request->aprobado;
            if($request->aprobado == 'SI') $notificar = true;
        }
        $sesion->update($params);

        $ids_form=array_column($request->tratamientos_vistas,'id');
        $ids_db=array_column($sesion->sesionTratamiento->toArray(),'id');

        $new_ids= array_diff($ids_form,$ids_db);
        $delete_ids= array_diff($ids_db,$ids_form);

        $sesion->cliente->update($request->only(['celular']));

        foreach ($new_ids as $value) {
            SesionTratamiento::create([
                'sesion_id'=>$sesion->id,
                'tratamiento_id'=>$value
            ]);
        }

        DB::table("sesiones_tratamientos")->where(['sesion_id'=>$sesion->id])->whereIn('tratamiento_id',$delete_ids)->delete();

        if($notificar){
            Mail::to($sesion->cliente->user->email)->send(new AvisoActivacionCita($sesion));
        }

        return response()->json([
            'data' => $sesion,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Sesion $sesion): JsonResponse
    {
        foreach($sesion->sesionTratamientoPuro as $tratamiento) {
            $tratamiento->delete();
        }

        foreach($sesion->valesSesiones as $vale) {
            $vale->delete();
        }

        foreach($sesion->ventasSesiones as $venta) {
            $venta->delete();
        }

        $sesion->delete();

        return response()->json(null, JsonResponse::HTTP_NO_CONTENT);
    }

    public function resumenFecha(Request $request): JsonResponse
    {
        $sesiones_resumen=[];
        if(!empty($request->fecha)){
            $sucursal_id = isset($request->sucursal_id) ? $request->sucursal_id : null;
            $cliente = isset($request->cliente) ? $request->cliente : null;
            $filtro_tratamiento = isset($request->filtro_tratamiento) ? $request->filtro_tratamiento : null;
            $filtro_terapista = isset($request->filtro_terapista) ? $request->filtro_terapista : null;

            $sql ="
            select
            distinct
            date_format(time,'%l %p') hora,
            time,
            te.name tera,
            count(distinct s.id) cantidad
            from
            (
                            select TIME('08:00:00') time union
                            select TIME('09:00:00') time union
                            select TIME('10:00:00') time union
                            select TIME('11:00:00') time union
                            select TIME('12:00:00') time union
                            select TIME('13:00:00') time union
                            select TIME('14:00:00') time union
                            select TIME('15:00:00') time union
                            select TIME('16:00:00') time union
                            select TIME('17:00:00') time union
                            select TIME('18:00:00') time union
                            select TIME('19:00:00') time
            ) as m
            INNER JOIN sesiones s on (
                    m.time between TIME(s.fecha_asignada) and (TIME(SUBTIME(ADDTIME(s.fecha_asignada,s.duracion),'00:00:01' )))
            )
            INNER JOIN clientes c on s.cliente_id=c.id
            INNER JOIN users te on s.terapista_id=te.id
            where date(s.fecha_asignada) = '".$request->fecha."'
            ";

            if (!$request->user()->hasRole('Cliente')) {
                $sql .= "
                    and s.aprobado='SI'
                ";
            } else {
                $sql .= "
                    and s.cliente_id=".$request->user()->cliente_id."
                ";
            }

            if ($sucursal_id != null && !empty($sucursal_id)) {
                $sql .= "
                    and s.sucursal_id=" . $sucursal_id . "
                ";
            }

            if ($cliente != null && !empty($cliente)) {
                $sql .= "
                    and lower(c.razonSocial) like lower('%" . $cliente . "%')
                ";
            }

            if ($filtro_terapista != null && !empty($filtro_terapista)) {
                $sql .= "
                    and s.terapista_id=" . $filtro_terapista . "
                ";
            }

            $sql.= "
                GROUP BY time, te.`name`
                ORDER BY m.time asc
            ";
            $consultas = DB::select($sql);
            $consulta_sessiones = Sesion::whereRaw("date(`fecha_asignada`)='{$request->fecha}'")
            ->with('cliente','terapista','sesionTratamientoPuro.tratamiento','sucursal')
            ->selectRaw("
            *,
            TIME(fecha_asignada) as tiempo_asignado_temp,
            (TIME(SUBTIME(ADDTIME(fecha_asignada,sesiones.`duracion`),'00:00:01' ))) as `time_final`,
            sesiones.`duracion` as `duracion_temp`, date_format(fecha_asignada,'%l:%i %p') as hora_inicio,
            date_format(TIME(SUBTIME(ADDTIME(fecha_asignada,sesiones.`duracion`),'00:00:01' )),'%l:%i %p') as hora_salida")
            ->when($request->sucursal_id,function (Builder $builder) use ($request) {
                $builder->Where('sucursal_id', $request->sucursal_id);
            })
            ->when($request->filtro_terapista,function (Builder $builder) use ($request) {
                $builder->Where('terapista_id', $request->filtro_terapista);
            })
            ->whereHas('cliente', function (Builder $builder) use ($request) {
                $builder->when($request->cliente,function (Builder $builder) use ($request) {
                    $builder->Where('razonSocial','like',"%{$request->cliente}%");
                });
            })
            ->when(!$request->user()->hasRole('Cliente'),function (Builder $builder) {
                $builder->Where('aprobado', 'SI');
            })
            ->when($request->user()->hasRole('Cliente'),function (Builder $builder) use ($request) {
                $builder->Where('cliente_id', $request->user()->cliente_id);
            })
            ->orderBy('fecha_asignada','asc')
            ->get();

            $horas = collect($consultas)->unique(function ($consulta) {
                return $consulta->time;
            });

            $horas->each(function ($hora) use ($consultas,$consulta_sessiones, &$sesiones_resumen) {
                $process = collect($consultas)->filter(function ($consulta) use ($hora) {
                    return $consulta->time == $hora->time;
                });

                $sesiones = collect($consulta_sessiones)->filter(function ($consulta) use ($hora) {
                    $hora_proceso= strtotime($hora->time);
                    $hora_inicio= strtotime($consulta->tiempo_asignado_temp);
                    $hora_final= strtotime($consulta->time_final);
                    return $hora_proceso >=$hora_inicio && $hora_proceso <=$hora_final;
                });
                $index=count($sesiones_resumen);
                $sesiones_resumen[$index]['grupo'] = $hora;
                $sesiones_resumen[$index]['grupo_cantidad'] = $process->sum('cantidad');
                $sesiones_resumen[$index]['ver'] = false;
                $sesiones_resumen[$index]['data'] = $process;
                $sesiones_resumen[$index]['sesiones'] = $sesiones;
            });


        }
        return response()->json([
            'data' =>[
                'sesiones_resumen'=>$sesiones_resumen,
                'fecha'=>fechaLetras($request->fecha),
            ]
        ], JsonResponse::HTTP_OK);
    }


    public function monitor(Request $request): JsonResponse
    {
        $sesiones_resumen=[];

        $sucursal_id = isset($request->sucursal_id) ? $request->sucursal_id : null;
        $fecha = isset($request->fecha) ? $request->fecha : '';

        $sql ="
        select
        distinct
        date_format(time,'%l %p') hora,
        time,
        te.name tera,
        count(distinct s.id) cantidad
        from
        (
                        select TIME('08:00:00') time union
                        select TIME('09:00:00') time union
                        select TIME('10:00:00') time union
                        select TIME('11:00:00') time union
                        select TIME('12:00:00') time union
                        select TIME('13:00:00') time union
                        select TIME('14:00:00') time union
                        select TIME('15:00:00') time union
                        select TIME('16:00:00') time union
                        select TIME('17:00:00') time union
                        select TIME('18:00:00') time union
                        select TIME('19:00:00') time
        ) as m
        INNER JOIN sesiones s on (
                m.time between TIME(s.fecha_asignada) and (TIME(SUBTIME(ADDTIME(s.fecha_asignada,s.duracion),'00:00:01' )))
        )
        INNER JOIN users te on s.terapista_id=te.id
        where
        (
            (
                s.fecha_asignada >= subtime(now() ,'00:30:00')
                and s.fecha_asignada <= concat(DATE(now()),' 23:59:59')
            )
            or
            (
                ADDTIME(s.fecha_asignada,s.duracion) >= addtime(now() ,'00:30:00')
                and ADDTIME(s.fecha_asignada,s.duracion) <= concat(DATE(now()),' 23:59:59')
            )
        )
        and time >= TIME(subtime(now() ,'00:30:00'))
        and s.aprobado='SI'
        ";

        if ($sucursal_id != null && !empty($sucursal_id)) {
            $sql .= "
                and s.sucursal_id=" . $sucursal_id . "
            ";
        }

        $sql.= "
            GROUP BY time, te.`name`
            ORDER BY m.time asc
        ";
        $consultas = DB::select($sql);
        // modificar aqui
        $consulta_sessiones = Sesion::whereRaw("
        (
            (
                fecha_asignada >= subtime(now() ,'00:30:00')
                and fecha_asignada <= concat(DATE(now()),' 23:59:59')
            )
            or
            (
                ADDTIME(fecha_asignada,duracion) >= addtime(now() ,'00:30:00')
                and ADDTIME(fecha_asignada,duracion) <= concat(DATE(now()),' 23:59:59')
            )
        )
        ")
        ->with('cliente','terapista','sesionTratamientoPuro.tratamiento','sucursal')
        ->selectRaw("
        *,
        TIME(fecha_asignada) as tiempo_asignado_temp,
        (TIME(SUBTIME(ADDTIME(fecha_asignada,sesiones.`duracion`),'00:00:01' ))) as `time_final`,
        sesiones.`duracion` as `duracion_temp`, date_format(fecha_asignada,'%l:%i %p') as hora_inicio,
        date_format(TIME(SUBTIME(ADDTIME(fecha_asignada,sesiones.`duracion`),'00:00:01' )),'%l:%i %p') as hora_salida")
        ->when($request->sucursal_id,function (Builder $builder) use ($request) {
            $builder->Where('sucursal_id', $request->sucursal_id);
        })
        ->where(['aprobado'=>'SI'])
        ->orderBy('fecha_asignada','asc')
        ->get();

        $horas = collect($consultas)->unique(function ($consulta) {
            return $consulta->time;
        });

        $horas->each(function ($hora) use ($consultas,$consulta_sessiones, &$sesiones_resumen) {
            $process = collect($consultas)->filter(function ($consulta) use ($hora) {
                return $consulta->time == $hora->time;
            });

            $sesiones = collect($consulta_sessiones)->filter(function ($consulta) use ($hora) {
                $hora_proceso= strtotime($hora->time);
                $hora_inicio= strtotime($consulta->tiempo_asignado_temp);
                $hora_final= strtotime($consulta->time_final);
                return $hora_proceso >=$hora_inicio && $hora_proceso <=$hora_final;
            });
            $index=count($sesiones_resumen);
            $sesiones_resumen[$index]['grupo'] = $hora;
            $sesiones_resumen[$index]['grupo_cantidad'] = $process->sum('cantidad');
            $sesiones_resumen[$index]['data'] = $process;
            $sesiones_resumen[$index]['sesiones'] = $sesiones;
        });

        return response()->json([
            'data' =>[
                'sesiones_resumen'=>$sesiones_resumen,
                'fecha'=>fechaLetras($fecha),
            ]
        ], JsonResponse::HTTP_OK);
    }

    public function getRevision(Request $request): JsonResponse
    {
        $sesiones= Sesion::leftjoin('clientes', 'sesiones.cliente_id', '=', 'clientes.id')
        ->leftjoin('sucursales', 'sesiones.sucursal_id', '=', 'sucursales.id')
        ->selectRaw("
            sesiones.*,
            date_format(sesiones.fecha_asignada,'%l %p') hora,
            date(sesiones.fecha_asignada) fecha,
            clientes.razonSocial cliente,
            sucursales.nombre sucursal
        ")
        ->when($request->cliente,function (Builder $builder) use ($request) {
            $builder->Where('clientes.razonSocial', 'like', "%{$request->cliente}%");
        })
        ->when($request->sucursal_id,function (Builder $builder) use ($request) {
            $builder->Where('sucursales.id', 'like', "%{$request->sucursal_id}%");
        })
        ->where(['aprobado'=>'NO'])
        ->get();
        return response()->json([
            'data' =>$sesiones,
            'cliente' =>$request->cliente,
        ], JsonResponse::HTTP_OK);
    }

    public function search(Request $request): JsonResponse
    {
        $sesion=[];
        if(!empty($request->search) || !empty($request->cliente_id)){
            $sesion=Sesion::leftjoin('clientes', 'clientes.id', '=', 'sesiones.cliente_id')
            ->leftjoin('sucursales', 'sucursales.id', '=', 'sesiones.sucursal_id')
            ->leftjoin('ventas_sesiones', 'ventas_sesiones.sesion_id', '=', 'sesiones.id')
            ->selectRaw("
                sesiones.*,
                date_format(sesiones.fecha_asignada,'%d-%m-%Y %l %p') fecha_asignada_custom,
                clientes.razonSocial cliente,
                sucursales.nombre sucursal
            ")
            ->when($request->search, function ($query) use ($request) {
                $query->where(function ($q) use ($request) {
                    $q->orWhere('clientes.razonSocial', 'LIKE', "%{$request->search}%")
                      ->orWhere('sucursales.nombre', 'LIKE', "%{$request->search}%")
                      ->orWhere('sesiones.fecha_asignada', 'LIKE', "%{$request->search}%")
                      ->orWhereRaw("date_format(sesiones.fecha_asignada,'%d-%m-%Y %l %p') LIKE '%{$request->search}%'");
                });

                // Si hay un cliente_id, filtramos dentro del grupo anterior
                if (!empty($request->cliente_id)) {
                    $query->where('sesiones.cliente_id', $request->cliente_id);
                }
            })
            ->when($request->sucursal_id, function ($query) use ($request) {
                $query->where('sesiones.sucursal_id', $request->sucursal_id);
            })
            ->when($request->cliente_id, function ($query) use ($request) {
                $query->where('sesiones.cliente_id', $request->cliente_id);
            })
            ->where(['clientes.estatus'=>'activo'])
            ->whereNull('ventas_sesiones.sesion_id')
            ->orderBy('sesiones.fecha_asignada','DESC')->get()->load('sesionTratamientoPrecio');
        }
        return response()->json([
            'data' =>$sesion,
        ], JsonResponse::HTTP_OK);
    }

    public function verificarVales(Request $request): JsonResponse
    {
        $request->validate([
            'uuid' => ['required_without:id'],
            'id' => ['required_without:uuid']
        ]);
        
        // Determinar si buscar por token (UUID) o por ID
        $whereClause = '';
        if ($request->has('uuid') && !empty($request->uuid)) {
            $whereClause = "token='".$request->uuid."'";
        } elseif ($request->has('id') && !empty($request->id)) {
            $whereClause = "vales.id='".$request->id."'";
        }
        
        $sql="
        SELECT vales.*, tratamientos.id tratamiento_id, tratamientos.tratamiento, tratamientos.precio precio_tratamiento, tratamientos.duracion duracion_tratamiento from vales
        inner join ventas_detalles on vales.id=ventas_detalles.vale_id
        inner join tratamientos on ventas_detalles.tratamiento_id=tratamientos.id
        LEFT JOIN sesiones on vales.id=sesiones.vale_id
        where ".$whereClause."
        and date(vigencia) >= date(now())
        and sesiones.vale_id is null
        and vales.estatus='activo'
        LIMIT 1
        ";
        $buscador = DB::select($sql);
        if(!isset($buscador[0])){
            return response()->json([
                'message' => 'Vale invalido',
                'errors' => [
                    'uuid' => [
                        'El vale es invalido motivos: no existe, no vigente, ya fue asignado'
                    ]
                ],
            ], JsonResponse::HTTP_UNPROCESSABLE_ENTITY);
        }
        return response()->json([
            'data' =>$buscador[0],
        ], JsonResponse::HTTP_OK);
    }
}
