<?php

namespace App\Http\Controllers;

use App\Models\VentaCorte;
use App\Models\Venta;
use App\Models\CorteCaja;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Database\Eloquent\Builder;

class CorteCajaController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(): JsonResponse
    {
        $corte=CorteCaja::where(['estatus'=>'activo'])->get();
        return response()->json([
            'data' =>$corte,
        ], JsonResponse::HTTP_OK);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'fecha_inicio' => ['required', 'date'],
            'fecha_fin' => ['required', 'date'],
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')],
            'total' => ['required', 'numeric', 'max:999999999.99'],
            'subtotal' => ['required', 'numeric', 'max:999999999.99'],
            'iva' => ['required', 'numeric', 'max:999999999.99'],
            'ieps' => ['required', 'numeric', 'max:999999999.99'],
            'total_efectivo' => ['required', 'numeric', 'max:999999999.99'],
            'total_tarjeta_debito' => ['required', 'numeric', 'max:999999999.99'],
            'total_tarjeta_credito' => ['required', 'numeric', 'max:999999999.99'],
            'total_tranferencia' => ['required', 'numeric', 'max:999999999.99'],
            'total_credito' => ['required', 'numeric', 'max:999999999.99'],
            'venta_id_temp' => ['required'],
        ]);

        $corte = CorteCaja::create($request->only([
            'fecha_inicio',
            'fecha_fin',
            'sucursal_id',
            'total',
            'subtotal',
            'iva',
            'ieps',
            'total_efectivo',
            'total_tarjeta_debito',
            'total_tarjeta_credito',
            'total_tranferencia',
            'total_credito',
        ]));

        foreach ($request->venta_id_temp as $key => $value) {
            VentaCorte::create([
                'corte_caja_id'=>$corte->id,
                'venta_id'=>$value
            ]);
        }

        return response()->json([
            'data' => $corte,
        ], JsonResponse::HTTP_OK);
    }


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(CorteCaja $corte): JsonResponse
    {
        $corte=CorteCaja::leftjoin('sucursales', 'sucursales.id', '=', 'cortes_cajas.sucursal_id')
        ->selectRaw("
            cortes_cajas.*,
            sucursales.nombre sucursal,
            date_format(cortes_cajas.fecha_inicio,'%d-%m-%Y %l %p') fecha_inicio_custom,
            date_format(cortes_cajas.fecha_fin,'%d-%m-%Y %l %p') fecha_fin_custom
        ")
        ->where(['cortes_cajas.id'=>$corte->id])->first();
        return response()->json([
            'data' => $corte->load('venta.venta'),
        ], JsonResponse::HTTP_OK);
    }

    public function paginacion(Request $request): JsonResponse
    {
        $cortes=CorteCaja::leftjoin('sucursales', 'sucursales.id', '=', 'cortes_cajas.sucursal_id')
        ->selectRaw("
            cortes_cajas.*,
            sucursales.nombre sucursal,
            date_format(cortes_cajas.fecha_inicio,'%d-%m-%Y %l %p') fecha_inicio_custom,
            date_format(cortes_cajas.fecha_fin,'%d-%m-%Y %l %p') fecha_fin_custom
        ")
        ->when($request->search,function (Builder $builder) use ($request) {
            $builder->orWhere('cortes_cajas.id', 'like', "%{$request->search}%")
                ->orWhere('sucursales.nombre', 'like', "%{$request->search}%")
                ->orWhereRaw("DATE_FORMAT(cortes_cajas.fecha_inicio, '%Y-%m-%d %l %p') LIKE ?", ["%{$request->search}%"])
                ->orWhereRaw("DATE_FORMAT(cortes_cajas.fecha_fin, '%Y-%m-%d %l %p') LIKE ?", ["%{$request->search}%"])
                ->orWhereRaw("DATE_FORMAT(cortes_cajas.fecha_inicio, '%d-%m-%Y %l %p') LIKE ?", ["%{$request->search}%"])
                ->orWhereRaw("DATE_FORMAT(cortes_cajas.fecha_fin, '%d-%m-%Y %l %p') LIKE ?", ["%{$request->search}%"])
                ->orWhere('cortes_cajas.total', 'like', "%{$request->search}%")
                ->orWhere('cortes_cajas.subtotal', 'like', "%{$request->search}%");
        })
        ->when($request->sucursal_id,function (Builder $builder) use ($request) {
            $builder->Where(['sucursales.id'=>$request->sucursal_id]);
        })
        ->where(['cortes_cajas.estatus'=>'activo'])->orderBy('updated_at','ASC')->paginate($request->perPage);
        return response()->json([
            'data' =>$cortes,
        ], JsonResponse::HTTP_OK);
    }

    public function calculoCorte(Request $request): JsonResponse
    {
        $request->validate([
            'fecha_inicio' => ['required', 'date'],
            'fecha_fin' => ['required', 'date'],
            'sucursal_id' => ['required', Rule::exists('sucursales', 'id')]
        ]);

        $cortes=Venta::leftjoin('sucursales', 'sucursales.id', '=', 'ventas.sucursal_id')
        ->leftjoin('pagos', 'pagos.venta_id', '=', 'ventas.id')
        ->leftjoin('ventas_cortes', 'ventas_cortes.venta_id', '=', 'ventas.id')
        ->selectRaw("
            ventas.sucursal_id,
            sucursales.nombre sucursal,
            sum(ventas.total) total,
            sum(ventas.subtotal) subtotal,
            sum(ventas.iva) iva,
            sum(
                case
                    when tipo_pago_id = 1 then pagos.total
                    else 0
                end
            ) total_efectivo,
            sum(
                case
                    when tipo_pago_id = 2 then pagos.total
                    else 0
                end
            ) total_tarjeta_debito,
            sum(
                case
                    when tipo_pago_id = 3 then pagos.total
                    else 0
                end
            ) total_tarjeta_credito,
            sum(
                case
                    when tipo_pago_id = 4 then pagos.total
                    else 0
                end
            ) total_credito,
            GROUP_CONCAT(ventas.id) venta_id_temp
        ")
        ->when($request->sucursal_id,function (Builder $builder) use ($request) {
            $builder->Where(['sucursales.id'=>$request->sucursal_id]);
        })
        ->where(['ventas.cancelado'=>'no'])
        ->where(['ventas.es_pago_tienda'=>'si'])
        ->whereNull('ventas_cortes.venta_id')
        ->whereBetween('ventas.fecha_registro',[$request->fecha_inicio,$request->fecha_fin])
        ->groupBy(['ventas.sucursal_id','sucursales.nombre'])
        ->first();

        $venta_id_temp=[];
        $ventasss=[332];
        if($cortes){
            $venta_id_temp=explode(",",$cortes->venta_id_temp);
            $ventasss=Venta::leftjoin('sucursales', 'sucursales.id', '=', 'ventas.sucursal_id')
            ->selectRaw("
                ventas.*,
                sucursales.nombre sucursal,
                date_format(ventas.fecha_registro,'%d-%m-%Y %l %p') fecha_asignada_custom
            ")
            ->whereRaw('ventas.id in('.$cortes->venta_id_temp.')')->get();
        }
        return response()->json([
            'data' =>[
                'cortes'=>$cortes,
                'venta_id_temp'=>$venta_id_temp,
                'ventas'=>$ventasss,
            ],
        ], JsonResponse::HTTP_OK);
    }
}
