<?php

namespace App\Http\Controllers;

use App\Models\Banco;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\Rule;

class BancoController extends Controller
{
    public function index(): JsonResponse
    {
        $bancos = Banco::where('estatus', 'activo')
            ->orderBy('nombre')
            ->get();

        return response()->json(['data' => $bancos]);
    }

    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'nombre' => ['required', 'string', 'max:255', 'unique:bancos,nombre'],
            'estatus' => ['sometimes', 'string', Rule::in(['activo', 'inactivo'])]
        ], [
            'nombre.required' => 'El nombre del banco es requerido',
            'nombre.unique' => 'Este banco ya existe en el sistema',
            'estatus.in' => 'El estatus debe ser activo o inactivo'
        ]);

        $banco = Banco::create($request->only(['nombre', 'estatus']));

        return response()->json([
            'data' => $banco,
            'message' => 'Banco creado exitosamente'
        ], JsonResponse::HTTP_CREATED);
    }

    public function show(Banco $banco): JsonResponse
    {
        return response()->json(['data' => $banco]);
    }

    public function update(Request $request, Banco $banco): JsonResponse
    {
        $request->validate([
            'nombre' => [
                'sometimes',
                'required',
                'string',
                'max:255',
                Rule::unique('bancos', 'nombre')->ignore($banco->id)
            ],
            'estatus' => ['sometimes', 'string', Rule::in(['activo', 'inactivo'])]
        ], [
            'nombre.required' => 'El nombre del banco es requerido',
            'nombre.unique' => 'Este banco ya existe en el sistema',
            'estatus.in' => 'El estatus debe ser activo o inactivo'
        ]);

        $banco->update($request->only(['nombre', 'estatus']));

        return response()->json([
            'data' => $banco,
            'message' => 'Banco actualizado exitosamente'
        ]);
    }

    public function destroy(Banco $banco): JsonResponse
    {
        // Opcionalmente, podrías querer verificar si el banco está en uso antes de eliminarlo
        $banco->update(['estatus' => 'inactivo']);

        return response()->json([
            'message' => 'Banco desactivado exitosamente'
        ]);
    }

    /**
     * Obtener lista de bancos para select
     */
    public function getForSelect(): JsonResponse
    {
        $bancos = Banco::where('estatus', 'activo')
            ->orderBy('nombre')
            ->select('id', 'nombre')
            ->get();

        return response()->json(['data' => $bancos]);
    }
}