<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Models\User;

return new class extends Migration
{
    /**
     * Run the migrations.
     * 
     * Sistema de Control de Lavado - Migración inicial de roles y usuarios
     * Crea los roles base: Administrador (Dueño), Recepcionista y Operador
     *
     * @return void
     */
    public function up()
    {
        // Crear permisos base del Sistema de Control de Lavado
        $permissions = [
            // Dashboard del sistema
            'dashboard.ver',
            
            // Módulo de Administración del Sistema de Control de Lavado
            'administracion.usuarios',
            'administracion.roles',
        ];

        foreach ($permissions as $permission) {
            Permission::firstOrCreate([
                'name' => $permission,
                'guard_name' => 'web'
            ]);
        }

        // Crear roles del Sistema de Control de Lavado según especificaciones:
        // - Administrador (Dueño): Acceso completo al sistema
        // - Recepcionista: Registro de órdenes, procesamiento de cobros, entrega final
        // - Operador: Operación en áreas de proceso (Lavado, Secado, Planchado)
        $adminRole = Role::create(['name' => 'Administrador', 'guard_name' => 'web']);
        $recepcionistaRole = Role::create(['name' => 'Recepcionista', 'guard_name' => 'web']);
        $operadorRole = Role::create(['name' => 'Operador', 'guard_name' => 'web']);

        // Asignar permisos al rol Administrador (todos los permisos base)
        $adminRole->givePermissionTo($permissions);

        // Los permisos específicos del módulo de lavandería se asignan en la migración 2026_02_09_000001_create_roles_lavanderia.php

        // Crear usuarios de prueba del Sistema de Control de Lavado
        // Nota: No se incluye 'estatus' aquí porque se agrega en una migración posterior (2024_01_15_000013)
        // El valor por defecto 'activo' se aplicará automáticamente cuando se agregue la columna
        $adminUser = User::create([
            'name' => 'Administrador',
            'email' => 'admin@gmail.com',
            'password' => '$2y$10$jOunoLNJ1KJ5MK3smYZ0q./c.PnQYnM4BAd0rikaRBAmGYveG8ydK',
            'email_verified_at' => now(),
        ]);

        $recepcionistaUser = User::create([
            'name' => 'Recepcionista',
            'email' => 'recepcionista@gmail.com',
            'password' => '$2y$10$jOunoLNJ1KJ5MK3smYZ0q./c.PnQYnM4BAd0rikaRBAmGYveG8ydK',
            'email_verified_at' => now(),
        ]);

        $operadorUser = User::create([
            'name' => 'Operador',
            'email' => 'operador@gmail.com',
            'password' => '$2y$10$jOunoLNJ1KJ5MK3smYZ0q./c.PnQYnM4BAd0rikaRBAmGYveG8ydK',
            'email_verified_at' => now(),
        ]);

        // Asignar roles a los usuarios
        $adminUser->assignRole($adminRole);
        $recepcionistaUser->assignRole($recepcionistaRole);
        $operadorUser->assignRole($operadorRole);
    }

    /**
     * Reverse the migrations.
     * 
     * Elimina los roles y usuarios del Sistema de Control de Lavado
     *
     * @return void
     */
    public function down()
    {
        // Eliminar usuarios de prueba del Sistema de Control de Lavado
        User::where('email', 'admin@gmail.com')->delete();
        User::where('email', 'recepcionista@gmail.com')->delete();
        User::where('email', 'operador@gmail.com')->delete();

        // Eliminar roles
        Role::where('name', 'Administrador')->delete();
        Role::where('name', 'Recepcionista')->delete();
        Role::where('name', 'Operador')->delete();

        // Eliminar permisos
        $permissions = [
            'dashboard.ver',
            'administracion.usuarios',
            'administracion.roles',
        ];
        
        Permission::whereIn('name', $permissions)->delete();
    }
};
